<?php
/**
 * Plugin Name: WP MultiControl – payment gateways sync
 * Description: master/client plugin to sync custom gateway snippets across multiple woocommerce sites.
 * Version: 1.1.1
 */

if (!defined('ABSPATH')) exit;

class WPMC_Gateways_Sync {
  const OPT_MODE     = 'wpmc_mode'; // master|client
  const OPT_SECRET   = 'wpmc_secret';
  const OPT_SITES    = 'wpmc_sites';
  const OPT_LAST_LOG = 'wpmc_last_log';

  const REST_NAMESPACE = 'wpmc/v1';
  const ROUTE_PING     = '/ping';
  const ROUTE_SYNC_ONE = '/sync-skrill-email';      // legacy
  const ROUTE_SYNC_ALL = '/sync-gateway-snippets';  // new

  public static function init() {
    add_action('admin_menu', [__CLASS__, 'admin_menu']);
    add_action('admin_init', [__CLASS__, 'handle_post']);
    add_action('rest_api_init', [__CLASS__, 'register_rest']);
  }

  public static function admin_menu() {
    add_menu_page('multicontrol', 'multicontrol', 'manage_options', 'wpmc', [__CLASS__, 'render_page'], 'dashicons-admin-site-alt3');
  }

  public static function render_page() {
    if (!current_user_can('manage_options')) return;

    $mode   = get_option(self::OPT_MODE, 'client');
    $secret = get_option(self::OPT_SECRET, '');
    $sites  = get_option(self::OPT_SITES, []);
    $log    = get_option(self::OPT_LAST_LOG, '');

    ?>
    <div class="wrap">
      <h1>wp multicontrol – payment gateways sync</h1>

      <form method="post">
        <?php wp_nonce_field('wpmc_save_settings'); ?>

        <h2>mode</h2>
        <p>
          <label><input type="radio" name="wpmc_mode" value="master" <?php checked($mode, 'master'); ?> /> master (push updates to other sites)</label><br/>
          <label><input type="radio" name="wpmc_mode" value="client" <?php checked($mode, 'client'); ?> /> client (receive updates from master)</label>
        </p>

        <h2>shared secret (hmac key)</h2>
        <p style="max-width: 900px;">
          <input type="text" name="wpmc_secret" value="<?php echo esc_attr($secret); ?>" style="width: 520px;" />
          <button class="button" name="wpmc_generate_secret" value="1">generate</button>
        </p>

        <hr/>

        <h2>master: sites list</h2>
        <p>only needed on the master site. add your other sites (clients) here.</p>

        <table class="widefat" style="max-width: 1000px;">
          <thead><tr><th>site name</th><th>base url</th><th>enabled</th></tr></thead>
          <tbody>
          <?php
            if (!is_array($sites)) $sites = [];
            if (empty($sites)) {
              echo '<tr><td colspan="3"><em>no sites yet. add one below.</em></td></tr>';
            } else {
              foreach ($sites as $i => $s) {
                $name = $s['name'] ?? '';
                $url  = $s['url'] ?? '';
                $en   = !empty($s['enabled']);
                ?>
                <tr>
                  <td><input type="text" name="wpmc_sites[<?php echo (int)$i; ?>][name]" value="<?php echo esc_attr($name); ?>" style="width: 260px;"></td>
                  <td><input type="text" name="wpmc_sites[<?php echo (int)$i; ?>][url]" value="<?php echo esc_attr($url); ?>" style="width: 520px;"></td>
                  <td><input type="checkbox" name="wpmc_sites[<?php echo (int)$i; ?>][enabled]" value="1" <?php checked($en); ?>></td>
                </tr>
                <?php
              }
            }
          ?>
          </tbody>
        </table>

        <p>
          <button class="button" name="wpmc_add_site" value="1">add another site row</button>
        </p>

        <hr/>

        <h2>sync values (fill only what you want to update)</h2>

        <table class="widefat" style="max-width: 1000px;">
          <thead><tr><th>gateway</th><th>field</th><th>value</th><th>used selector</th></tr></thead>
          <tbody>
            <tr>
              <td>skrill</td>
              <td>email</td>
              <td><input type="text" name="wpmc_skrill_email" placeholder="email@example.com" style="width: 520px;"></td>
              <td><code>p.skrill_email_custom &gt; strong</code></td>
            </tr>
            <tr>
              <td>neteller</td>
              <td>email</td>
              <td><input type="text" name="wpmc_neteller_email" placeholder="email@example.com" style="width: 520px;"></td>
              <td><code>p.neteller_email_custom (and strong)</code></td>
            </tr>
            <tr>
              <td>webmoney</td>
              <td>account</td>
              <td><input type="text" name="wpmc_webmoney_account" placeholder="z652279978313" style="width: 520px;"></td>
              <td><code>p.webmoney_email_custom &gt; strong</code></td>
            </tr>
            <tr>
              <td>yoomoney</td>
              <td>link</td>
              <td><input type="text" name="wpmc_yoomoney_link" placeholder="https://yoomoney.ru/to/..." style="width: 520px;"></td>
              <td><code>p.yoomoney_email_custom &gt; a</code></td>
            </tr>
            <tr>
              <td>revolut/volet</td>
              <td>account</td>
              <td><input type="text" name="wpmc_volet_account" placeholder="u 6222 0406 5755" style="width: 520px;"></td>
              <td><code>p.volet_email_custom &gt; strong</code></td>
            </tr>
            <tr>
              <td>alipay</td>
              <td>email</td>
              <td><input type="text" name="wpmc_alipay_email" placeholder="email@gmail.com" style="width: 520px;"></td>
              <td><code>p.alipay_email_custom</code></td>
            </tr>
            <tr>
              <td>payeer</td>
              <td>account id</td>
              <td><input type="text" name="wpmc_payeer_account" placeholder="p1095513603" style="width: 520px;"></td>
              <td><code>p.payeer_accnu_custom</code></td>
            </tr>
          </tbody>
        </table>

        <p style="margin-top: 14px;">
          <button class="button button-primary" name="wpmc_push_multi" value="1">push to all sites</button>
          <button class="button" name="wpmc_test" value="1">test connection</button>
          <button class="button" name="wpmc_save" value="1">save settings</button>
        </p>
      </form>

      <hr/>
      <h2>last log</h2>
      <pre style="background:#fff;border:1px solid #ccd0d4;padding:12px;white-space:pre-wrap;"><?php echo esc_html($log ?: 'no log yet.'); ?></pre>

      <hr/>
      <h2>client endpoints</h2>
      <p><strong>ping (get):</strong> <code><?php echo esc_html(home_url('/wp-json/' . self::REST_NAMESPACE . self::ROUTE_PING)); ?></code></p>
      <p><strong>sync (post):</strong> <code><?php echo esc_html(home_url('/wp-json/' . self::REST_NAMESPACE . self::ROUTE_SYNC_ALL)); ?></code></p>
    </div>
    <?php
  }

  public static function handle_post() {
    if (!current_user_can('manage_options')) return;
    if (empty($_POST)) return;
    if (!check_admin_referer('wpmc_save_settings')) return;

    if (!empty($_POST['wpmc_generate_secret'])) {
      update_option(self::OPT_SECRET, wp_generate_password(40, true, true));
      self::log('generated new shared secret.');
      wp_safe_redirect(admin_url('admin.php?page=wpmc'));
      exit;
    }

    if (!empty($_POST['wpmc_add_site'])) {
      $sites = get_option(self::OPT_SITES, []);
      if (!is_array($sites)) $sites = [];
      $sites[] = ['name' => '', 'url' => '', 'enabled' => 1];
      update_option(self::OPT_SITES, $sites);
      self::log('added a new site row.');
      wp_safe_redirect(admin_url('admin.php?page=wpmc'));
      exit;
    }

    if (!empty($_POST['wpmc_save'])) {
      $mode = (isset($_POST['wpmc_mode']) && $_POST['wpmc_mode'] === 'master') ? 'master' : 'client';
      $secret = isset($_POST['wpmc_secret']) ? sanitize_text_field($_POST['wpmc_secret']) : '';
      $sites = (isset($_POST['wpmc_sites']) && is_array($_POST['wpmc_sites'])) ? self::sanitize_sites($_POST['wpmc_sites']) : get_option(self::OPT_SITES, []);

      update_option(self::OPT_MODE, $mode);
      update_option(self::OPT_SECRET, $secret);
      update_option(self::OPT_SITES, $sites);

      self::log('saved settings.');
      wp_safe_redirect(admin_url('admin.php?page=wpmc'));
      exit;
    }

    if (!empty($_POST['wpmc_test'])) {
      $mode = get_option(self::OPT_MODE, 'client');
      if ($mode !== 'master') {
        self::log('test blocked: site is not in master mode.');
        wp_safe_redirect(admin_url('admin.php?page=wpmc'));
        exit;
      }
      $sites = get_option(self::OPT_SITES, []);
      self::log(self::test_sites($sites));
      wp_safe_redirect(admin_url('admin.php?page=wpmc'));
      exit;
    }

    if (!empty($_POST['wpmc_push_multi'])) {
      $mode = get_option(self::OPT_MODE, 'client');
      if ($mode !== 'master') {
        self::log('push blocked: site is not in master mode.');
        wp_safe_redirect(admin_url('admin.php?page=wpmc'));
        exit;
      }

      $secret = get_option(self::OPT_SECRET, '');
      if (empty($secret)) {
        self::log('push failed: shared secret is empty.');
        wp_safe_redirect(admin_url('admin.php?page=wpmc'));
        exit;
      }

      $payload = [
        'skrill_email'      => sanitize_email($_POST['wpmc_skrill_email'] ?? ''),
        'neteller_email'    => sanitize_email($_POST['wpmc_neteller_email'] ?? ''),
        'webmoney_account'  => sanitize_text_field($_POST['wpmc_webmoney_account'] ?? ''),
        'yoomoney_link'     => esc_url_raw(trim($_POST['wpmc_yoomoney_link'] ?? '')),
        'volet_account'     => sanitize_text_field($_POST['wpmc_volet_account'] ?? ''),
        'alipay_email'      => sanitize_email($_POST['wpmc_alipay_email'] ?? ''),
        'payeer_account'    => sanitize_text_field($_POST['wpmc_payeer_account'] ?? ''),
      ];

      $payload = array_filter($payload, fn($v) => $v !== '' && $v !== null);

      if (empty($payload)) {
        self::log('push skipped: no values filled.');
        wp_safe_redirect(admin_url('admin.php?page=wpmc'));
        exit;
      }

      $sites = get_option(self::OPT_SITES, []);
      self::log(self::push_multi_to_sites($sites, $secret, $payload));
      wp_safe_redirect(admin_url('admin.php?page=wpmc'));
      exit;
    }
  }

  private static function sanitize_sites(array $raw): array {
    $out = [];
    foreach ($raw as $row) {
      $name = sanitize_text_field($row['name'] ?? '');
      $url  = esc_url_raw(trim($row['url'] ?? ''));
      $en   = !empty($row['enabled']) ? 1 : 0;
      if ($url !== '') $out[] = ['name' => $name, 'url' => rtrim($url, '/'), 'enabled' => $en];
    }
    return $out;
  }

  private static function test_sites(array $sites): string {
    $lines = [];
    $lines[] = 'test start ' . gmdate('c');

    $enabled = array_values(array_filter((array)$sites, fn($s) => !empty($s['enabled']) && !empty($s['url'])));
    if (empty($enabled)) {
      $lines[] = 'no enabled sites found.';
      return implode("\n", $lines);
    }

    foreach ($enabled as $s) {
      $name = $s['name'] ?: $s['url'];
      $base = rtrim($s['url'], '/');
      $endpoint = $base . '/wp-json/' . self::REST_NAMESPACE . self::ROUTE_PING;

      $resp = wp_remote_get($endpoint, ['timeout' => 20]);
      if (is_wp_error($resp)) {
        $lines[] = "❌ {$name}: " . $resp->get_error_message();
        continue;
      }

      $code = wp_remote_retrieve_response_code($resp);
      $body = wp_remote_retrieve_body($resp);

      if ($code >= 200 && $code < 300) $lines[] = "✅ {$name}: ok ({$code}) " . self::shorten($body);
      else $lines[] = "❌ {$name}: http {$code} " . self::shorten($body);
    }

    $lines[] = 'test end';
    return implode("\n", $lines);
  }

  private static function push_multi_to_sites(array $sites, string $secret, array $values): string {
    $lines = [];
    $lines[] = 'push start ' . gmdate('c');
    $lines[] = 'values: ' . wp_json_encode($values);

    $enabled = array_values(array_filter((array)$sites, fn($s) => !empty($s['enabled']) && !empty($s['url'])));
    if (empty($enabled)) {
      $lines[] = 'no enabled sites found.';
      return implode("\n", $lines);
    }

    foreach ($enabled as $s) {
      $name = $s['name'] ?: $s['url'];
      $base = rtrim($s['url'], '/');
      $endpoint = $base . '/wp-json/' . self::REST_NAMESPACE . self::ROUTE_SYNC_ALL;

      $nonce = wp_generate_password(18, false, false);
      $ts = time();

      $body = ['values' => $values, 'ts' => $ts, 'nonce' => $nonce];
      $json = wp_json_encode($body);
      $sig  = hash_hmac('sha256', $json, $secret);

      $resp = wp_remote_post($endpoint, [
        'timeout' => 20,
        'headers' => ['Content-Type' => 'application/json', 'X-WPMC-Signature' => $sig],
        'body' => $json,
      ]);

      if (is_wp_error($resp)) {
        $lines[] = "❌ {$name}: " . $resp->get_error_message();
        continue;
      }

      $code  = wp_remote_retrieve_response_code($resp);
      $rbody = wp_remote_retrieve_body($resp);

      if ($code >= 200 && $code < 300) $lines[] = "✅ {$name}: ok ({$code}) " . self::shorten($rbody);
      else $lines[] = "❌ {$name}: http {$code} " . self::shorten($rbody);
    }

    $lines[] = 'push end';
    return implode("\n", $lines);
  }

  private static function shorten(string $s, int $max = 260): string {
    $s = trim(wp_strip_all_tags($s));
    return (strlen($s) > $max) ? substr($s, 0, $max) . '...' : $s;
  }

  private static function log(string $msg) {
    update_option(self::OPT_LAST_LOG, $msg);
  }

  public static function register_rest() {
    register_rest_route(self::REST_NAMESPACE, self::ROUTE_PING, [
      'methods'  => 'GET',
      'callback' => [__CLASS__, 'rest_ping'],
      'permission_callback' => '__return_true',
    ]);

    register_rest_route(self::REST_NAMESPACE, self::ROUTE_SYNC_ONE, [
      'methods'  => 'POST',
      'callback' => [__CLASS__, 'rest_sync_skrill_email'],
      'permission_callback' => '__return_true',
    ]);

    register_rest_route(self::REST_NAMESPACE, self::ROUTE_SYNC_ALL, [
      'methods'  => 'POST',
      'callback' => [__CLASS__, 'rest_sync_gateway_snippets'],
      'permission_callback' => '__return_true',
    ]);
  }

  public static function rest_ping(\WP_REST_Request $req) {
    return new \WP_REST_Response([
      'ok' => true,
      'plugin' => 'wpmc-gateways-sync',
      'version' => '1.1.1',
      'mode' => get_option(self::OPT_MODE, 'client'),
    ], 200);
  }

  private static function verify_signed_request(\WP_REST_Request $req) {
    $mode = get_option(self::OPT_MODE, 'client');
    if ($mode !== 'client') return ['ok' => false, 'status' => 403, 'error' => 'not_in_client_mode'];

    $secret = get_option(self::OPT_SECRET, '');
    if (empty($secret)) return ['ok' => false, 'status' => 403, 'error' => 'missing_secret'];

    $raw = $req->get_body();
    $sig = isset($_SERVER['HTTP_X_WPMC_SIGNATURE']) ? trim($_SERVER['HTTP_X_WPMC_SIGNATURE']) : '';
    if (empty($sig) || !hash_equals(hash_hmac('sha256', $raw, $secret), $sig)) {
      return ['ok' => false, 'status' => 403, 'error' => 'bad_signature'];
    }

    $data = json_decode($raw, true);
    if (!is_array($data)) return ['ok' => false, 'status' => 400, 'error' => 'bad_json'];

    $ts    = (int)($data['ts'] ?? 0);
    $nonce = sanitize_text_field($data['nonce'] ?? '');

    if ($ts <= 0 || abs(time() - $ts) > 300) return ['ok' => false, 'status' => 403, 'error' => 'timestamp_out_of_range'];

    $nonce_key = 'wpmc_nonce_' . md5($nonce);
    if ($nonce && get_transient($nonce_key)) return ['ok' => false, 'status' => 403, 'error' => 'replay_detected'];
    if ($nonce) set_transient($nonce_key, 1, 10 * MINUTE_IN_SECONDS);

    return ['ok' => true, 'data' => $data];
  }

  public static function rest_sync_skrill_email(\WP_REST_Request $req) {
    $v = self::verify_signed_request($req);
    if (!$v['ok']) return new \WP_REST_Response(['ok' => false, 'error' => $v['error']], $v['status']);

    $data = $v['data'];
    $email = sanitize_email($data['email'] ?? '');
    if (empty($email) || !is_email($email)) return new \WP_REST_Response(['ok' => false, 'error' => 'invalid_email'], 400);

    $result = self::apply_values_locally(['skrill_email' => $email]);
    return new \WP_REST_Response(['ok' => true] + $result, 200);
  }

  public static function rest_sync_gateway_snippets(\WP_REST_Request $req) {
    $v = self::verify_signed_request($req);
    if (!$v['ok']) return new \WP_REST_Response(['ok' => false, 'error' => $v['error']], $v['status']);

    $data = $v['data'];
    $values = $data['values'] ?? null;
    if (!is_array($values) || empty($values)) return new \WP_REST_Response(['ok' => false, 'error' => 'missing_values'], 400);

    $clean = [];
    if (!empty($values['skrill_email']) && is_email($values['skrill_email'])) $clean['skrill_email'] = sanitize_email($values['skrill_email']);
    if (!empty($values['neteller_email']) && is_email($values['neteller_email'])) $clean['neteller_email'] = sanitize_email($values['neteller_email']);
    if (!empty($values['webmoney_account'])) $clean['webmoney_account'] = sanitize_text_field($values['webmoney_account']);
    if (!empty($values['yoomoney_link'])) $clean['yoomoney_link'] = esc_url_raw(trim($values['yoomoney_link']));
    if (!empty($values['volet_account'])) $clean['volet_account'] = sanitize_text_field($values['volet_account']);
    if (!empty($values['alipay_email']) && is_email($values['alipay_email'])) $clean['alipay_email'] = sanitize_email($values['alipay_email']);
    if (!empty($values['payeer_account'])) $clean['payeer_account'] = sanitize_text_field($values['payeer_account']);

    if (empty($clean)) return new \WP_REST_Response(['ok' => false, 'error' => 'no_valid_values'], 400);

    $result = self::apply_values_locally($clean);
    return new \WP_REST_Response(['ok' => true] + $result, 200);
  }

  private static function apply_values_locally(array $values): array {
    global $wpdb;

    $rows = $wpdb->get_results("SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE 'woocommerce\_%\_settings'", ARRAY_A);

    $updated_options = [];
    $matched_options = [];
    $counts = [
      'skrill' => 0,
      'neteller' => 0,
      'webmoney' => 0,
      'yoomoney' => 0,
      'volet' => 0,
      'alipay' => 0,
      'payeer' => 0,
    ];

    foreach ($rows as $r) {
      $opt = $r['option_name'];
      $val = get_option($opt);
      if (!is_array($val)) continue;

      $desc  = (string)($val['description'] ?? '');
      $instr = (string)($val['instructions'] ?? '');

      $hay = $desc . ' ' . $instr;

      $has_marker =
        (strpos($hay, 'skrill_email_custom') !== false) ||
        (strpos($hay, 'neteller_email_custom') !== false) ||
        (strpos($hay, 'webmoney_email_custom') !== false) ||
        (strpos($hay, 'yoomoney_email_custom') !== false) ||
        (strpos($hay, 'volet_email_custom') !== false) ||
        (strpos($hay, 'alipay_email_custom') !== false) ||
        (strpos($hay, 'payeer_accnu_custom') !== false);

      if (!$has_marker) continue;

      $matched_options[] = $opt;
      $changed = false;

      if ($desc !== '') {
        $newDesc = $desc;

        // skrill
        if (!empty($values['skrill_email']) && strpos($newDesc, 'skrill_email_custom') !== false) {
          $pattern = '/(<p\s+class=["\']skrill_email_custom["\']\s*>\s*<strong>)(.*?)(<\/strong>\s*<\/p>)/is';
          $repl = '$1' . esc_html($values['skrill_email']) . '$3';
          $tmp = preg_replace($pattern, $repl, $newDesc, -1, $replaced);
          if ($tmp !== null && $replaced > 0) { $newDesc = $tmp; $counts['skrill'] += $replaced; $changed = true; }
        }

        // neteller: in your screenshot it is <strong><p class="neteller_email_custom">email</p></strong>
        if (!empty($values['neteller_email']) && strpos($newDesc, 'neteller_email_custom') !== false) {
          $pattern = '/(<p\s+class=["\']neteller_email_custom["\']\s*>)(.*?)(<\/p>)/is';
          $repl = '$1' . esc_html($values['neteller_email']) . '$3';
          $tmp = preg_replace($pattern, $repl, $newDesc, -1, $replaced);
          if ($tmp !== null && $replaced > 0) { $newDesc = $tmp; $counts['neteller'] += $replaced; $changed = true; }
        }

        // webmoney
        if (!empty($values['webmoney_account']) && strpos($newDesc, 'webmoney_email_custom') !== false) {
          $pattern = '/(<p\s+class=["\']webmoney_email_custom["\']\s*>\s*<strong>\s*)(.*?)(\s*<\/strong>\s*<\/p>)/is';
          $repl = '$1' . esc_html($values['webmoney_account']) . '$3';
          $tmp = preg_replace($pattern, $repl, $newDesc, -1, $replaced);
          if ($tmp !== null && $replaced > 0) { $newDesc = $tmp; $counts['webmoney'] += $replaced; $changed = true; }
        }

        // yoomoney
        if (!empty($values['yoomoney_link']) && strpos($newDesc, 'yoomoney_email_custom') !== false) {
          $pattern = '/(<p\s+class=["\']yoomoney_email_custom["\']\s*>\s*<a\s+href=["\'])([^"\']*)(["\']\s*>)(.*?)(<\/a>\s*<\/p>)/is';
          $repl = '$1' . esc_url($values['yoomoney_link']) . '$3' . esc_html($values['yoomoney_link']) . '$5';
          $tmp = preg_replace($pattern, $repl, $newDesc, -1, $replaced);
          if ($tmp !== null && $replaced > 0) { $newDesc = $tmp; $counts['yoomoney'] += $replaced; $changed = true; }
        }

        // volet/revolut
        if (!empty($values['volet_account']) && strpos($newDesc, 'volet_email_custom') !== false) {
          $pattern = '/(<p\s+class=["\']volet_email_custom["\']\s*>\s*<strong>)(.*?)(<\/strong>\s*<\/p>)/is';
          $repl = '$1' . esc_html($values['volet_account']) . '$3';
          $tmp = preg_replace($pattern, $repl, $newDesc, -1, $replaced);
          if ($tmp !== null && $replaced > 0) { $newDesc = $tmp; $counts['volet'] += $replaced; $changed = true; }
        }

        // alipay
        if (!empty($values['alipay_email']) && strpos($newDesc, 'alipay_email_custom') !== false) {
          $pattern = '/(<p\s+class=["\']alipay_email_custom["\']\s*>)(.*?)(<\/p>)/is';
          $repl = '$1' . esc_html($values['alipay_email']) . '$3';
          $tmp = preg_replace($pattern, $repl, $newDesc, -1, $replaced);
          if ($tmp !== null && $replaced > 0) { $newDesc = $tmp; $counts['alipay'] += $replaced; $changed = true; }
        }

        // payeer
        if (!empty($values['payeer_account']) && strpos($newDesc, 'payeer_accnu_custom') !== false) {
          $pattern = '/(<p\s+class=["\']payeer_accnu_custom["\']\s*>)(.*?)(<\/p>)/is';
          $repl = '$1' . esc_html($values['payeer_account']) . '$3';
          $tmp = preg_replace($pattern, $repl, $newDesc, -1, $replaced);
          if ($tmp !== null && $replaced > 0) { $newDesc = $tmp; $counts['payeer'] += $replaced; $changed = true; }
        }

        if ($newDesc !== $desc) {
          $val['description'] = $newDesc;
          $changed = true;
        }
      }

      // optional email replace in instructions
      if ($instr !== '') {
        $newInstr = $instr;
        if (!empty($values['skrill_email'])) {
          $newInstr = preg_replace('/[A-Z0-9._%+\-]+@[A-Z0-9.\-]+\.[A-Z]{2,}/i', $values['skrill_email'], $newInstr);
        }
        if (!empty($values['neteller_email'])) {
          $newInstr = preg_replace('/[A-Z0-9._%+\-]+@[A-Z0-9.\-]+\.[A-Z]{2,}/i', $values['neteller_email'], $newInstr);
        }
        if (!empty($values['alipay_email'])) {
          $newInstr = preg_replace('/[A-Z0-9._%+\-]+@[A-Z0-9.\-]+\.[A-Z]{2,}/i', $values['alipay_email'], $newInstr);
        }
        if ($newInstr !== $instr) {
          $val['instructions'] = $newInstr;
          $changed = true;
        }
      }

      if ($changed) {
        update_option($opt, $val);
        $updated_options[] = $opt;
      }
    }

    return [
      'matched_options' => array_values(array_unique($matched_options)),
      'updated_options' => array_values(array_unique($updated_options)),
      'counts' => $counts,
      'updated_count' => count(array_unique($updated_options)),
    ];
  }
}

WPMC_Gateways_Sync::init();
